/*****************************************************************************
 *   timer.c: Timer C file for Philips LPC29xx Family Microprocessors
 *
 *   Copyright(C) 2007, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2007.09.01  ver 1.00    Preliminary version, first Release
 *
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 *****************************************************************************/

#include "LPC29xx.h"			/* LPC29xx Peripheral Registers	*/
#include "type.h"
#include "target.h"
#include "irq.h"
#include "timer.h"
#include "dma.h"


extern volatile FOC_STRUCT FOC;	
volatile DWORD timer_counter = 0;

/******************************************************************************
** Function name:		delayMs
**
** Descriptions:		Initialize timer, set timer interval, reset timer,
**						Delay X milo second based on the parameter
**
** parameters:			Delay interval
** Returned value:		None
** 
******************************************************************************/
void delayMs( DWORD Interval ) 
{
  TIM0_TCR = 1<<1;			// Reset counter 
  TIM0_TCR &= ~(1<<1);
  
  /* divided by 80 to make it 1 second interval */
  /* please note, the PERIPH_CLK could be different for different
  peripherals, the UART clock is the same as SYS_CLK 80Mhz. */
  TIM0_PR = (PERIPH_CLK / Interval) - 1;
  TIM0_TCR |= (1<<0);		// Start timer 
  while ( TIM0_TC >= TIM0_PR );
  TIM0_TCR &= ~(1<<0);		// Stop timer 
  return;
}

/******************************************************************************
** Function name:		Timer0Handler
**
** Descriptions:		Timer/Counter 0 interrupt handler
**
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void TIMER_ISR( void )
{
  DWORD regVal;
   
  timer_counter++;		// Global timer counter 

  regVal = TIM0_INT_STATUS;
  TIM0_INT_CLR_STATUS = regVal;
  return;
}

/******************************************************************************
** Function name:		Timer_Enable
**
** Descriptions:		Enable timer
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void Timer_Enable( void )
{
  TIM0_TCR = (1<<1);
  TIM0_TCR &= ~(1<<1);
  TIM0_TCR |= (1<<0);
  TIM0_INT_SET_ENABLE = 0x01;
  return;
}

/******************************************************************************
** Function name:		Timer_Disable
**
** Descriptions:		Disable timer
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void Timer_Disable( void )
{
  TIM0_TCR = 1<<1;
  TIM0_TCR &= ~(1<<1);
  TIM0_TCR &= ~(1<<0);
  TIM0_INT_CLR_ENABLE = 0x01;
  return;
}

/******************************************************************************
** Function name:		Timer_Init
**
** Descriptions:		Initialize timer, set timer interval, reset timer,
**						install timer interrupt handler
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void Timer_Init( DWORD timer_interval ) 
{
  timer_counter = 0;

  /* Reset counter */
  TIM0_TCR = 1<<1;
  TIM0_TCR &= ~(1<<1);   
 
  /* divided by 80 to make it 1 second interval */
  /* please note, the PERIPH_CLK could be different for different
  peripherals, the UART clock is the same as SYS_CLK 80Mhz. */
  TIM0_PR = (PERIPH_CLK/1000000); 

  /* Reset counter if Match 0 counter equals to timer counter. */
  TIM0_MCR = 0x01;

  /* Set MATCH0 register */
  TIM0_MR0 = timer_interval-1;

  install_irq( TIMER0_INT, (void *)TIMER_ISR, LOWEST_PRIORITY );
  EnableIntReq_IRQ( TIMER0_INT, ACTIVE_HIGH, LOWEST_PRIORITY );
  return;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
